<?php

// Copyright (c) 2016 Interfacelab LLC. All rights reserved.
//
// Released under the GPLv3 license
// http://www.gnu.org/licenses/gpl-3.0.html
//
// **********************************************************************
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
// **********************************************************************

namespace MediaCloud\Plugin\Tools\Video\Driver\Mux;

use Elementor\Elements_Manager;
use Elementor\Plugin;
use MediaCloud\Plugin\Tasks\TaskManager;
use MediaCloud\Plugin\Tools\Video\Driver\Mux\Data\MuxDatabase;
use MediaCloud\Plugin\Tools\Video\Driver\Mux\Elementor\MuxVideoWidget;
use MediaCloud\Plugin\Tools\Video\Driver\Mux\Models\MuxAsset;
use MediaCloud\Plugin\Tools\Video\Driver\Mux\Tasks\MigrateToMuxTask;
use MediaCloud\Plugin\Tools\Tool;
use MediaCloud\Plugin\Tools\ToolsManager;
use MediaCloud\Plugin\Utilities\Logging\Logger;
use MediaCloud\Plugin\Utilities\View;
use function MediaCloud\Plugin\Utilities\anyEmpty;
use function MediaCloud\Plugin\Utilities\arrayPath;

class MuxTool extends Tool {
	/** @var null|MuxToolSettings|MuxToolProSettings */
	protected $settings = null;

	/** @var MuxHooks */
	protected $hooks = null;

	/** @var MuxShortcode|null  */
	protected $shortCode = null;

	public function __construct($toolName, $toolInfo, $toolManager) {
	
		$this->settings = MuxToolProSettings::instance();
		

		add_action('media-cloud/tools/register-setting-type', [$this, 'registerMuxSettingTypes'], 10, 5);

		parent::__construct($toolName, $toolInfo, $toolManager);

		$this->initBlocks();
	}

	//region Tool Overrides
	public function enabled() {
		$enabled = parent::enabled();

		if (empty($this->settings->tokenID) || empty($this->settings->tokenSecret)) {
			return false;
		}

		if (empty($this->settings->webhookSecret)) {
			return false;
		}

		return $enabled;
	}

	public function hasSettings() {
		return true;
	}

	public function setup() {
		if ($this->enabled()) {
			MuxDatabase::init();

			$this->shortCode = new MuxShortcode();
			$this->hooks = new MuxHooks();

			
			TaskManager::registerTask(MigrateToMuxTask::class);
			


			if (is_admin()) {
				add_action('admin_enqueue_scripts', function(){
					wp_enqueue_script('mux-admin-js', ILAB_PUB_JS_URL.'/mux-admin.js', null, null, true);
					wp_enqueue_style('mux-admin-css', ILAB_PUB_CSS_URL . '/mux-admin.css' );
				});

				$this->integrateWithMediaLibrary();
				$this->integrateWithAdmin();
			}


			static::enqueuePlayer(is_admin());
		}
	}
	//endregion

	//region Properties
	/**
	 * @return MuxHooks
	 */
	public function hooks() {
		return $this->hooks;
	}
	//endregion

	//region Static
	public static function enqueuePlayer($admin = false) {
		
			add_action((!empty($admin)) ? 'admin_enqueue_scripts' : 'wp_enqueue_scripts', function() {
				/** @var MuxToolProSettings $settings */
				$settings = MuxToolProSettings::instance();

				if($settings->playerType === 'videojs') {
					$lastDep = 'mux_videojs';
					if(!empty($settings->envKey)) {
						wp_enqueue_script('mux_videojs', ILAB_PUB_JS_URL . '/videojs-player-data.js', null, null, true);
					} else {
						wp_enqueue_script('mux_videojs', ILAB_PUB_JS_URL . '/videojs-player.js', null, null, true);
					}

					$playerSettings = ['plugins' => []];

					if($settings->playerAllowAirPlay) {
						$playerSettings['plugins']['airPlay'] = ['addButtonToControlBar' => true];
					}

					if($settings->playerAllowQualitySelect) {
						$playerSettings['plugins']['httpSourceSelector'] = ['default' => 'auto'];
					}

					$lastDep = apply_filters('media-cloud/video/videojs-enqueue-plugins', $lastDep);

					if (empty($settings->playerOverrideNative)) {
						$playerSettings['html5'] = [
							'vhs' => [
								'overrideNative' => false
							],
							'nativeAudioTracks' => true,
							'nativeVideoTracks' => true,
						];
					} else {
						$playerSettings['html5'] = [
							'vhs' => [
								'overrideNative' => true
							],
							'nativeAudioTracks' => false,
							'nativeVideoTracks' => false,
						];
					}

					$playerSettings = apply_filters('media-cloud/video/videojs-player-settings', $playerSettings);

					$deps = empty($lastDep) ? null: [$lastDep];
					wp_enqueue_script('mux_video_player_videojs', ILAB_PUB_JS_URL . '/mux-player.js', $deps, null, true);
					wp_localize_script('mux_video_player_videojs', 'muxPlayerSettings', $playerSettings);

					wp_enqueue_style('mux_video_player_style', ILAB_PUB_CSS_URL . '/mux-player.css', null, null);
					do_action('media-cloud/video/videojs-enqueue-style', 'mux_video_player_style');
				} else if($settings->playerType === 'hlsjs') {
					wp_enqueue_script('mux_video_player_hlsjs', ILAB_PUB_JS_URL . '/mux-hls.js', null, null, true);
				}
			});
		
	}
	//endregion

	// region Settings
	public function registerMuxSettingTypes($option, $optionInfo, $group, $groupInfo, $conditions) {
		if ($optionInfo['type'] === 'mux-webhook') {
			$description = arrayPath($optionInfo,'description',null);

			add_settings_field($option,
				$optionInfo['title'],
				[$this,'renderWebhookField'],
				$this->options_page,
				$group,
				['option'=>$option, 'description'=>$description, 'conditions' => $conditions]);
		}
	}

	public function renderWebhookField($args) {
		echo View::render_view('settings.fields.mux-webhook',[
			'value' => home_url('/__mux/webhook'),
			'name' => $args['option'],
			'conditions' => $args['conditions'],
			'description' => (isset($args['description'])) ? $args['description'] : false
		]);
	}


	public function providerHelp() {
		return [
			'mux' => [
				[ 'title' => 'Sign Up For Mux Account', 'external_url' => 'https://mux.com' ],
				[ 'title' => 'Read Documentation', 'url' => 'https://support.mediacloud.press/articles/documentation/video-encoding/about-video-encoding' ],
			]
		];
	}
	//endregion

	//region Blocks
	protected function initBlocks() {
		add_action('init', function() {
			wp_register_style('mux_video_block_style', ILAB_BLOCKS_URL.'mediacloud-mux.blocks.style.css', ['wp-editor'], null);
			wp_register_script('mux_video_block_js',ILAB_BLOCKS_URL.'mediacloud-mux.blocks.js', ['wp-blocks', 'wp-i18n', 'wp-element', 'wp-editor'], null, true);
			wp_register_style('mux_video_block_editor_style', ILAB_BLOCKS_URL.'mediacloud-mux.blocks.editor.css', ['wp-edit-blocks'], null);

			register_block_type('media-cloud/mux-video-block', [
				'style'         => 'mux_video_block_style',
				'editor_script' => 'mux_video_block_js',
				'editor_style'  => 'mux_video_block_editor_style',
			]);

		});

		add_filter('block_categories', function($categories, $post) {
			foreach($categories as $category) {
				if ($category['slug'] === 'mediacloud') {
					return $categories;
				}
			}

			$categories[] = [
				'slug' => 'mediacloud',
				'title' => 'Media Cloud',
				'icon' => null
			];

			return $categories;
		}, 10, 2);


		if (class_exists('Elementor\Plugin')) {
			add_action('elementor/widgets/widgets_registered', function() {
				Plugin::instance()->widgets_manager->register_widget_type(new MuxVideoWidget());
			});

			add_action('elementor/elements/categories_registered', function($elementsManager) {
				/** @var Elements_Manager $elementsManager */
				$elementsManager->add_category('media-cloud', [
					'title' => 'Media Cloud',
					'icon' => 'fa fa-plug'
				]);
			}, 10, 1);

			add_filter('the_content', function($content) {
				return MuxVideoWidget::filterContent($content);
			}, PHP_INT_MAX, 1);

			add_action('wp_enqueue_scripts', function() {
				wp_enqueue_style('mcloud-elementor', trailingslashit(ILAB_PUB_CSS_URL).'mcloud-elementor.css', [], MEDIA_CLOUD_VERSION);
			});
		}
	}
	//endregion

	//region Integration
	protected function handleCaptionDelete() {
		$nonce = arrayPath($_POST, 'nonce');
		if (empty($nonce) || !wp_verify_nonce($nonce, 'mux-delete-caption')) {
			wp_send_json(['status' => 'error', 'message' => 'Missing nonce.'], 400);
		}

		$aid = (int)arrayPath($_POST, 'aid', null);
		$trackId = arrayPath($_POST, 'trackId', null);
		if (anyEmpty($aid, $trackId)) {
			wp_send_json(['status' => 'error', 'message' => 'Missing parameters.'], 400);
		}

		/** @var MuxAsset $asset */
		$asset = MuxAsset::instance($aid);
		if (empty($asset)) {
			wp_send_json(['status' => 'error', 'message' => 'Invalid asset ID.'], 400);
		}

		if ($asset->deleteCaptions($trackId)) {
			wp_send_json(['status' => 'ok'], 200);
		}

		wp_send_json(['status' => 'error', 'message' => 'Unknown error.'], 400);
	}

	protected function handleCaptionUpload() {
		$nonce = arrayPath($_POST, 'nonce');
		if (empty($nonce) || !wp_verify_nonce($nonce, 'mux-upload-caption')) {
			wp_send_json(['status' => 'error', 'message' => 'Missing nonce.'], 400);
		}

		$aid = (int)arrayPath($_POST, 'aid', null);
		if (empty($aid)) {
			wp_send_json(['status' => 'error', 'message' => 'Missing asset ID.'], 400);
		}

		$asset = MuxAsset::instance($aid);
		if (empty($asset)) {
			wp_send_json(['status' => 'error', 'message' => 'Invalid asset ID.'], 400);
		}

		$language = arrayPath($_POST, 'language', null);
		if (empty($language)) {
			wp_send_json(['status' => 'error', 'message' => 'Missing language.'], 400);
		}

		$cc = !empty((int)arrayPath($_POST, 'cc', null));

		$finfo = new \finfo(FILEINFO_MIME);
		$info = $finfo->file($_FILES['file']['tmp_name']);
		$infoParts = explode(';', $info);
		$mimeType = array_shift($infoParts);
		if (!in_array($mimeType, ['text/plain', ' text/vtt', 'text/srt'])) {
			Logger::error("Invalid captions mime type: $mimeType", [], __METHOD__, __LINE__);
			wp_send_json(['status' => 'error', 'message' => 'Invalid file type'], 400);
		}

		$uploadedFile = wp_upload_bits($_FILES['file']['name'], null, file_get_contents($_FILES['file']['tmp_name']));
		if (isset($uploadedFile['error']) && !empty($uploadedFile['error'])) {
			Logger::error("Error importing caption: {$uploadedFile['error']}", [], __METHOD__, __LINE__);
			wp_send_json(['status' => 'error', 'message' => 'Error importing captions'], 400);
		}

		if ($asset->addCaptions($language, $uploadedFile['url'], $cc)) {
			wp_send_json(['status' => 'ok'], 200);
		}

		wp_send_json(['status' => 'error', 'message' => 'Unknown error.'], 400);
	}

	protected function integrateWithAdmin() {
		if (current_user_can('manage_options')) {
			add_action('wp_ajax_mux-upload-caption', function() {
				$this->handleCaptionUpload();
			});

			add_action('wp_ajax_mux-delete-caption', function() {
				$this->handleCaptionDelete();
			});
		}

		add_action('admin_init', function() {
			add_meta_box('mcloud-mux-meta', 'Mux Info', function($post) {
				/** @var \WP_Post $post */
				$asset = MuxAsset::assetForAttachment($post->ID);
				echo View::render_view('admin.mux-properties', ['asset' => $asset]);
			}, 'attachment', 'side', 'low');
		});
	}

	protected function integrateWithMediaLibrary() {
		add_filter('wp_prepare_attachment_for_js', [$this, 'prepareAttachmentForJS'], 1000, 3);

		if (ToolsManager::instance()->toolEnabled('storage')) {
			add_filter('media-cloud/media-library/attachment-classes', function($additionalClasses) {
				$additionalClasses = '<# if (data.hasOwnProperty("mux")) {#>has-mux mux-status-{{data.mux.status}}<#}#>'.$additionalClasses;
				return $additionalClasses;
			});

			add_filter('media-cloud/media-library/attachment-icons', function($additionalIcons) {
				$muxIcon = '<i class="mux-status-icon"></i>';
				return $muxIcon.$additionalIcons;
			});
		} else {
			$this->hookMediaLibraryGrid();
		}


		if ($this->settings->deleteFromMux) {
			add_action('delete_attachment', [$this, 'deleteAttachment'], 999);
		}
	}

	private function hookMediaLibraryGrid() {
		add_action('admin_footer', function() {
			?>
			<script>
                jQuery(document).ready(function () {
                    var attachTemplate = jQuery('#tmpl-attachment');
                    if (attachTemplate) {
                        var txt = attachTemplate.text();

                        var search = '<div class="attachment-preview js--select-attachment type-{{ data.type }} subtype-{{ data.subtype }} {{ data.orientation }}">';
                        var replace = '<div class="attachment-preview js--select-attachment type-{{ data.type }} subtype-{{ data.subtype }} {{ data.orientation }} <# if (data.hasOwnProperty("mux")) {#>has-mux mux-status-{{data.mux.status}}<#}#>"><i class="mux-status-icon"></i>';
                        txt = txt.replace(search, replace);
                        attachTemplate.text(txt);
                    }

                    var attachTemplate = jQuery('#tmpl-attachment-grid-view');
                    if (attachTemplate) {
                        var txt = attachTemplate.text();

                        var search = '<div class="attachment-preview js--select-attachment type-{{ data.type }} subtype-{{ data.subtype }} {{ data.orientation }}">';
                        var replace = '<div class="attachment-preview js--select-attachment type-{{ data.type }} subtype-{{ data.subtype }} {{ data.orientation }} <# if (data.hasOwnProperty("mux")) {#>has-mux mux-status-{{data.mux.status}}<#}#>"><i class="mux-status-icon"></i>';
                        txt = txt.replace(search, replace);
                        attachTemplate.text(txt);
                    }
                });
			</script>
			<?php
		});
	}

	/**
	 * Filters the attachment data prepared for JavaScript. (https://core.trac.wordpress.org/browser/tags/4.8/src/wp-includes/media.php#L3279)
	 *
	 * @param array $response
	 * @param int|object $attachment
	 * @param array $meta
	 *
	 * @return array
	 */
	public function prepareAttachmentForJS($response, $attachment, $meta) {
		if(empty($meta) || !isset($meta['mux'])) {
			return $response;
		}

		$mux = $meta['mux'];
		if (!empty($_REQUEST['post_id'])) {
			try {
				$asset = MuxAsset::asset($mux['muxId']);
				if ($asset === null) {
					return $response;
				}

				$mux['src'] = $asset->videoUrl(false);
				$mux['gif'] = $asset->gifUrl(false);
			} catch (\Exception $ex) {
				Logger::error("Mux: Exception fetching Mux Asset {$mux['muxId']}: ".$ex->getMessage(), [], __METHOD__, __LINE__);
				return $response;
			}
		}

		$response['mux'] = $mux;

		return $response;
	}

	public function deleteAttachment($id) {
		if (!$this->settings->deleteFromMux) {
			return $id;
		}

		$data = wp_get_attachment_metadata($id);
		$muxId = arrayPath($data, 'mux/muxId', null);
		if (empty($muxId)) {
			return $id;
		}

		$asset = MuxAsset::asset($muxId);
		if ($asset === null) {
			return $asset;
		}

		try {
			MuxAPI::assetAPI()->deleteAsset($muxId);
		} catch (\Exception $ex) {
			Logger::error('Mux: Error deleting asset from Mux: '.$ex->getMessage(), [], __METHOD__, __LINE__);
		}
		$asset->delete();

		return $id;
	}
	//endregion
}